/*
Copyright (C) 2007-2024 Victor Matei Petrescu

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <SDL.h>
#include <time.h>

#include "dyn/sde.h"

#include "../graphics/graphics.h"
#include "config.h"

#include "defstrans.h"
#include "readfile.h"
#include "game.h"


void dtextb(char *text,int x,int y,float col){
  unsigned char c;
  setcolor(0,0,0);
  dtext(text,x-1,y-1,0); dtext(text,x+1,y+1,0); dtext(text,x-1,y+1,0); dtext(text,x+1,y-1,0);
  if(col<0.001){col=0.001;}else{if(col>1.999){col=1.999;}}
  if(col<=1.0){
    setcolrnb(0.5*col);
  }else{
    c=(unsigned char)(255*(col-1.0));
    setcolor(c,c,c);
  }
  dtext(text,x,y,0);
}


/*print time*/
void tmformat(float tm,char *s){
  int hr,mn,sc;
  hr=(int)(tm/3600.0);
  mn=(int)(tm/60.0-hr*60.0);
  sc=(int)(tm-mn*60.0-hr*3600.0);
  sprintf(s,"%2d:%2d:%2d",hr,mn,sc);
  if(hr<10){s[0]='0';}
  if(mn<10){s[3]='0';}
  if(sc<10){s[6]='0';}
}


/*generate 2d model for arrow*/
void genmodsw(){
  float l,h;

l=30.0; h=0.2887*l;
addmod2d();
addpoint2d(-0.5*l,-h); addpoint2d(0.5*l,-h); addpoint2d(0.0,2.0*h);
addpoint2d(-0.2*l,-h); addpoint2d(0.2*l,-h);
addpoint2d(-0.2*l,-l); addpoint2d(0.2*l,-l);
addtriangle2d(0,1,2,0.3); addtriangle2d(3,4,6,0.3); addtriangle2d(3,5,6,0.3);
addline2d(0,2); addline2d(1,2); addline2d(0,3); addline2d(4,1);
addline2d(3,5); addline2d(5,6); addline2d(6,4);
addmod2d();
addpoint2d(-0.5*l,-h); addpoint2d(0.5*l,-h); addpoint2d(0.0,2.0*h);
addpoint2d(-0.2*l,-h); addpoint2d(0.2*l,-h);
addpoint2d(-0.2*l,-l); addpoint2d(0.2*l,-l);
addtriangle2d(0,1,2,0.1); addtriangle2d(3,4,6,0.1); addtriangle2d(3,5,6,0.1);
addline2d(0,2); addline2d(1,2); addline2d(0,3); addline2d(4,1);
addline2d(3,5); addline2d(5,6); addline2d(6,4);
}


/*initialize random numbers*/
void inrnd(){
  int i,t;
t=time(NULL); if(t<0){t=-t;} t=t%16384;
for(i=1;i<=t;i++){rand();}
}


int frand01(float min,float max){
  int r;
r=rand(); r%=100000; max+=0.9;
return (int)(min+(float)r*1.0e-5f*(max-min));
}


int main(){
  char text[64],**carf=NULL,**trackf=NULL,sel;
  int ncars=0,ntracks=0,ccar=0,ctrack=0,w,screenwidth=640,screenheight=480,
    fullscr=0,
    i,quit=0,
    t0frame, /*t0frame - moment when image starts to be displayed*/
    nob,nto,camflag=2, /*number of objects and of object types*/
    turn, /*-1 left, 0 no turn, 1 right*/
    dmode, /*1 forward, -1 reverse*/
    rocket, /*1 on, 0 off*/
    nstepsf, /*number of simulation steps/frame*/
    showcrd=0;

  SDL_Event event;

  float fov=90.0,
    zfog=200,zmax=400,tframe=0,xan=0,/*tframe-time necessary for display; xan-number of displayed images*/
    timp,dstr,timerep=0, /*total time, distance traveled, time for replay*/
    vrx,vrxmax,vrxmr, /*rot. speed*/
    arx,arxmax,arxmr, /*rot. acceleration*/
    vrot3, /*rot. speed of cat. 3 objects*/
    vrotc,vrcmax,rotc, /*rot. speed and rotation of camera*/
    realstep, /*real time step (s)*/
    speed,dspeed,rotspeed,acc,
    af=0,bf=0,hbf=0, /*acceleration and brake factors*/
    tr,trmax,
    *pos;

  gameobj *objs=NULL; /*objects(=NULL so no warning)*/

  vhc car; /*vehicle*/

#if REPLAY==1
rpglob.t=NULL;
rpglob.p=NULL;
#endif

readtok("config","screen:fov:zmax:fullscr:cars:tracks");

while((w=tkgetword())){
  switch(w){
    case 1: screenwidth=tkgetint(); screenheight=tkgetint();
            break;
    case 2: fov=tkgetfloat();
            break;
    case 3: zmax=tkgetfloat(); zfog=0.5*zmax;
            break;
    case 4: fullscr=tkgetint();
            break;
    case 5: ncars=tkgetint();
            if(!(carf=(char **)malloc(ncars*sizeof(char *)))){printf("Out of memory\r\n"); exit(1);}
            for(i=0;i<ncars;i++){
              if(!(carf[i]=(char *)malloc(32*sizeof(char)))){printf("Out of memory\r\n"); exit(1);}
              tkgetstring(carf[i]);
            }
            break;
    case 6: ntracks=tkgetint();
            if(!(trackf=(char **)malloc(ntracks*sizeof(char *)))){printf("Out of memory\r\n"); exit(1);}
            for(i=0;i<ntracks;i++){
              if(!(trackf[i]=(char *)malloc(32*sizeof(char)))){printf("Out of memory\r\n"); exit(1);}
              tkgetstring(trackf[i]);
            }
            break;
    default: break;
  }
}


if(screenwidth<640){screenwidth=640;}
if(screenheight<400){screenheight=400;}
initgraph(screenwidth,screenheight,fullscr,SIMCVER);
genmodsw();
setbgcolor(0,0,0); clearscreen();

inrnd(); ccar=frand01(0,ncars-1); ctrack=frand01(0,ntracks-1);

STARTGAME:

sel='0';
while(sel!='1'){
  setchrsize(4,4); setcolor(255,100,100); setpos(0,322);
  printw("\r\n %s",SIMCVER);
  setchrsize(3,3); printw("\r\n\r\n");
  setcolor(255,255,255);
  printw(" 1. Start driving\r\n 2. Change car (%d)  \r\n 3. Change track (%d)  \r\n 4. Replay\r\n 5. Quit",ccar+1,ctrack+1);
  setcolor(255,255,50); setpos(0,100);
  printw(" Select option (1...5):\r\n #"); scanw("%c",&sel);
  setpos(0,100); printw("                        \r\n                        ");
  setpos(0,52); printw("                        ");
  switch(sel){
    case '0': setcolor(255,200,100); setpos(0,52);
            printw(" Select '5' to quit");
            break;
    case '2': setcolor(100,255,255); setpos(0,100);
            printw(" Car (1...%d):\r\n #",ncars); scanw("%d",&ccar);
            setpos(0,100); printw("                        \r\n                        ");
            if(ccar<1){ccar=1;}
            if(ccar>ncars){ccar=ncars;}
            ccar--;
            break;
    case '3': setcolor(100,255,255); setpos(0,100);
            printw(" Track (1...%d):\r\n #",ntracks); scanw("%d",&ctrack);
            setpos(0,100); printw("                        \r\n                        ");
            if(ctrack<1){ctrack=1;}
            if(ctrack>ntracks){ctrack=ntracks;}
            ctrack--;
            break;
    case '4': if(timerep>0.1){goto SHOWREPLAY;}else{goto STARTGAME;}
    case '5': goto ENDGAME;
    default: break;
  }
}

quit=0;

initgraph3d(fov,0.1,zmax);

initSDE();
setGravity(-9.81,0.0,0.0);

strcpy(text,trackf[ctrack]); /*text=file name*/
objs=readtrack(text,objs,&nto,&nob); /*read objects from file*/

strcpy(text,carf[ccar]);
objs=readvehicle(text,objs,&nto,&nob,&car); /*read vehicle from file*/

#if REPLAY==1
initreplay(ccar,ctrack,car.nob);
#endif

tr=0; /*rocket running time*/
trmax=car.rocket[1]; /*time until fuel exhausted*/

setgraph3dobjs(objs,0,nob);

vrx=0; arx=0;
vrxmr=vrxmax=0.36;
arxmr=arxmax=vrxmax/1.5;
turn=0;
dmode=1;
rocket=0;
vrot3=1.1;
vrcmax=0.79;
vrotc=0;
rotc=0;

timp=0.0; dstr=0.0; xan=0; /*pornit cronometru*/
tframe=0.5; /*assuming 2 frames/second*/


while(!quit){

/*t0frame=clock();*/
t0frame=SDL_GetTicks();
xan++;

if(speed<10){vrxmr=vrxmax; arxmr=arxmax;}
else{
  vrxmr=vrxmax/(0.1*speed);
  arxmr=arxmax/(0.1*speed);
}

switch(turn){
  case 0: if(vrx>0){arx=-arxmr*1.5;}else{if(vrx<0){arx=arxmr*1.5;}else{arx=0;}}
          if(fabs(vrx)<2.25*tframe*arx){arx=0; vrx=0;}
          break;
  case -1: if(vrx>-vrxmr){arx=-arxmr; if(vrx>0){arx*=1.5;}}else{arx=0;}
           break;
  case 1: if(vrx<vrxmr){arx=arxmr; if(vrx<0){arx*=1.5;}}else{arx=0;}
          break;
  default: break;
}

vrx+=arx*tframe;
if(vrx>vrxmr){vrx=vrxmr;}
if(vrx<-vrxmr){vrx=-vrxmr;}

/*simulation*/
nstepsf=(int)(tframe/STIMESTEP)+1; /*number of simulation steps/frame*/
realstep=tframe/nstepsf; /*simulation time step*/

speed=0.1/realstep; /*decrease simulation speed if < 10fps*/
if(nstepsf>(int)speed){nstepsf=(int)speed;}

if(((rotspeed*car.accel)>car.power)&&(fabs(af)>1.0e-3)){
  af=dmode*(car.power/(rotspeed*car.accel));
}

if(rocket){
  tr+=tframe;
  if(tr>(trmax-0.002)){tr=trmax-0.001; rocket=0;}
}

runsteps(objs,&car,realstep,nstepsf,vrx,af,bf,hbf,rocket);

for(i=0;i<nob;i++){
  if(objs[i].lev==3){
    rotab(&objs[i],objs[i].vx[0],objs[i].vy[0],objs[i].vz[0],objs[i].vx[3],objs[i].vy[3],objs[i].vz[3],vrot3*tframe);
    setgraph3dobjs(objs,i,i+1);
  }
}

rdspeed(&car,&speed,&rotspeed,&dspeed);
acc=dspeed/tframe;

#if GAMESOUND==1
setvoice(0,4.0*rotspeed,0.75); setvoice(1,6.0*rotspeed,0.75); setvoice(2,5.33*rotspeed,0.75);

if(acc>20.0){
  if(acc>25.0){
    setvoice(3,1.0,0.75);
  }else{
    setvoice(3,1.0,0.35);
  }
}else{
    setvoice(3,1.0,0.0);
}

if(rocket){
  setvoice(3,1.0,1.0);
}
#endif

rotc+=vrotc*tframe; if(camflag!=1){rotc=0; vrotc=0;}
setcamg(&car,camflag,rotc,objs);

setgraph3dobjs(objs,nob-(car.nob),nob);

drawobj3all();
drawfog3d(zfog,zmax);

setchrsize(2,2);
tmformat(timp,text); dtextb(text,7,screenheight-23,2.0);
sprintf(text,"%1.2f km",0.001*dstr); dtextb(text,7,screenheight-75,2.0);
setchrsize(4,4);
sprintf(text,"%1.0f km/h",speed*3.6); dtextb(text,7,screenheight-57,2.0);

switch(dmode){
  case 1: setcolor(255,255,255); drawmod2d(0,screenwidth-58,49,1.0,0.0);
          dtextb("F",screenwidth-36,20,2.0);
          break;
  case -1: setcolor(255,255,255); drawmod2d(1,screenwidth-58,23,1.0,3.1416);
           dtextb("R",screenwidth-36,20,2.0);
           break;
  default: break;
}

setchrsize(3,3);
if(rocket){
  dtextb("Rocket:",8,8,2.0);
}else{
  dtextb("Rocket:",8,8,1.5);
}
setchrsize(4,4);
sprintf(text,"%1.0f%%",100.0*(trmax-tr)/trmax);
if(rocket){
  dtextb(text,156,7,0.9*(trmax-tr)/trmax);
}else{
  dtextb(text,156,7,1.7);
}

if(showcrd==1){
  pos=DGLOBpart[car.bid[0]].pos;
  setchrsize(2,2);
  sprintf(text,"X: %1.1f",pos[0]); dtextb(text,screenwidth-164,screenheight-23,2.0);
  sprintf(text,"Y: %1.1f",pos[1]); dtextb(text,screenwidth-164,screenheight-41,2.0);
  sprintf(text,"Z: %1.1f",pos[2]); dtextb(text,screenwidth-164,screenheight-59,2.0);
}

display();

dstr+=(speed*tframe);


while(SDL_PollEvent(&event)){
switch(event.type){

case SDL_KEYDOWN:
  switch(event.key.keysym.sym){
    case SDLK_q:
    case SDLK_UP: af=dmode;
         break;
    case SDLK_a:
    case SDLK_DOWN: bf=1;
         break;
    case SDLK_SPACE: hbf=1;
         break;
    case SDLK_o:
    case SDLK_LEFT: turn=-1;
         break;
    case SDLK_p:
    case SDLK_RIGHT: turn=1;
         break;

    case SDLK_r: dmode=-dmode;
         break;

    case SDLK_s: rocket=1;
         break;

    case SDLK_c: camflag++; if(camflag>3){camflag=1;}
         rotc=0; vrotc=0;
         break;

    case SDLK_n: vrotc=-vrcmax;
         break;

    case SDLK_m: vrotc=vrcmax;
         break;

    case SDLK_x: showcrd=1-showcrd;
         break;

    case SDLK_ESCAPE: quit=1;

    default: break;
  } break;
case SDL_KEYUP:
  switch(event.key.keysym.sym){
    case SDLK_q:
    case SDLK_UP: af=0;
         break;
    case SDLK_a:
    case SDLK_DOWN: bf=0;
         break;
    case SDLK_SPACE: hbf=0;
         break;
    case SDLK_o:
    case SDLK_LEFT: if(turn==-1){turn=0;}
         break;
    case SDLK_p:
    case SDLK_RIGHT: if(turn==1){turn=0;}
         break;

    case SDLK_r: af=0;
         break;

    case SDLK_s: rocket=0;
         break;

    case SDLK_n: vrotc=0;
         break;

    case SDLK_m: vrotc=0;
         break;

    default: break;
  } break;

case SDL_QUIT: quit=1;

default: break;
}
}
SDL_Delay(2);
tframe=0.001*(float)(SDL_GetTicks()-t0frame);
timp+=tframe;
}

quitSDE();

for(i=0;i<nto;i++){
  free(refglob[i].p1); free(refglob[i].p2); free(refglob[i].p3);
}

free(refglob); free(objs);
closegraph3d();

for(i=0;i<4;i++){setvoice(i,100.0,0.0);}

setchrsize(2,2); setbgcolor(0,0,0); setcolor(200,200,150); setpos(0,402); clearscreen();
printw("--------------------------------");
printw("\r\n %1.2f km, %1.2f seconds\r\n",dstr/1000,timp);
printw(" average speed: %1.2f km/h\r\n",3.6*dstr/timp);
printw(" average framerate: %1.2f f/s\r\n",xan/timp);
printw("--------------------------------\r\n");

timerep=timp; /*for replay*/

goto STARTGAME;


SHOWREPLAY:

quit=0;

initgraph3d(fov,0.1,zmax);

initSDE();

ccar=rpglob.car; ctrack=rpglob.track;

strcpy(text,trackf[ctrack]);
objs=readtrack(text,objs,&nto,&nob);

strcpy(text,carf[ccar]);
objs=readvehicle(text,objs,&nto,&nob,&car);

setgraph3dobjs(objs,0,nob);


vrot3=1.1;
vrcmax=0.79;
vrotc=0;
rotc=0;

timp=0.0; dstr=0.0; xan=0;
tframe=0.5;


while(!quit){

t0frame=SDL_GetTicks();
xan++;

readrep(objs,&car,timp);

for(i=0;i<nob;i++){
  if(objs[i].lev==3){
    rotab(&objs[i],objs[i].vx[0],objs[i].vy[0],objs[i].vz[0],objs[i].vx[3],objs[i].vy[3],objs[i].vz[3],vrot3*tframe);
    setgraph3dobjs(objs,i,i+1);
  }
}

rotc+=vrotc*tframe; if(camflag!=1){rotc=0; vrotc=0;}
setcamg(&car,camflag,rotc,objs);

setgraph3dobjs(objs,nob-(car.nob),nob);

drawobj3all();
drawfog3d(zfog,zmax);

setchrsize(2,2);
 if (timp>=timerep){tmformat(timerep,text);}
 else{tmformat(timp,text);}
dtextb(text,7,screenheight-23,2.0);

setchrsize(3,3);

dtextb("Replay",8,screenheight-50,2.0);

if(showcrd==1){
  pos=DGLOBpart[car.bid[0]].pos;
  setchrsize(2,2);
  sprintf(text,"X: %1.1f",pos[0]); dtextb(text,screenwidth-164,screenheight-23,2.0);
  sprintf(text,"Y: %1.1f",pos[1]); dtextb(text,screenwidth-164,screenheight-41,2.0);
  sprintf(text,"Z: %1.1f",pos[2]); dtextb(text,screenwidth-164,screenheight-59,2.0);
}

display();

if(timp<=timerep){dstr+=(speed*tframe);}

while(SDL_PollEvent(&event)){
switch(event.type){

case SDL_KEYDOWN:
  switch(event.key.keysym.sym){
    case SDLK_c: camflag++; if(camflag>3){camflag=1;}
         rotc=0; vrotc=0;
         break;

    case SDLK_n: vrotc=-vrcmax;
         break;

    case SDLK_m: vrotc=vrcmax;
         break;

    case SDLK_x: showcrd=1-showcrd;
         break;

    case SDLK_ESCAPE: quit=1;

    default: break;
  } break;
case SDL_KEYUP:
  switch(event.key.keysym.sym){
    case SDLK_n: vrotc=0;
         break;

    case SDLK_m: vrotc=0;
         break;

    default: break;
  } break;

case SDL_QUIT: quit=1;

default: break;
}
}
SDL_Delay(2);
tframe=0.001*(float)(SDL_GetTicks()-t0frame);
timp+=tframe;
}

quitSDE();

for(i=0;i<nto;i++){
  free(refglob[i].p1); free(refglob[i].p2); free(refglob[i].p3);
}

free(refglob); free(objs);
closegraph3d();

for(i=0;i<4;i++){setvoice(i,100.0,0.0);}

setchrsize(2,2); setbgcolor(0,0,0); setcolor(200,200,150); setpos(0,402); clearscreen();

goto STARTGAME;


ENDGAME:

freereplay();

freemod2d();
closegraph();

for(i=0;i<ncars;i++){free(carf[i]);}
free(carf);
for(i=0;i<ntracks;i++){free(trackf[i]);}
free(trackf);

freetok();

return 0;
}
