/*
Copyright (C) 2007-2018 Victor Matei Petrescu

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "grconfig.h"
#include "graphics.h"


#if SDLVER==1

  SDL_Event RGLOB_event;
  SDL_Surface *RGLOB_screen;

  #if SOUND==1

    void my_audio_callback(void *userdata, unsigned char *stream, int len);

    /* Open the audio device */
    SDL_AudioSpec *RGLOB_desired, *RGLOB_obtained;
    float RGLOB_userdata[20];

  #endif

#elif SDLVER==2

  SDL_Event RGLOB_event;
  SDL_Surface *RGLOB_screen;
  SDL_Window *RGLOB_window;

  #if SOUND==1

    void my_audio_callback(void *userdata, unsigned char *stream, int len);

    /* Open the audio device */
    SDL_AudioSpec *RGLOB_desired, *RGLOB_obtained;
    float RGLOB_userdata[20];

  #endif

#else

  Uint32 *RGLOB_screen;
  unsigned char RGLOB_bmp[54]={66,77,54,16,14,0,0,0,0,0,54,0,0,0,40,0,0,0,128,2,0,0,224,1,0,0,1,0,24,0,0,0,0,0,0,16,14,0,19,11,0,0,19,11,0,0,0,0,0,0,0,0,0,0};
  /*start of bmp file;width:RGLOB_bmp[19],[18];height:[23],[22]*/
  #undef SOUND
  #define SOUND 0

#endif


unsigned int RGLOB_width,RGLOB_height,RGLOB_bpp; /*bpp-bytes per pixel*/
void **RGLOB_lpinc,**RGLOB_lpdec; /*[i]-pointer to the start of line i*/
char RGLOB_initgr=0,RGLOB_csize[2]; /*initgr=1-graphics initialized; 0-not,csize-character size*/
Uint32 RGLOB_color[2];
int RGLOB_xcr,RGLOB_ycr; /*current coordinates*/
Uint32 *RGLOB_chrset; /*characters*/


typedef struct _mod2d{
  unsigned int np[2]; /*np[0]-nr.of points;np[1]-nr.points for which memory was allocated*/
  float *x; /*x[2i],x[2i+1]-coord.x and y of point i*/
  unsigned int nl[2]; /*nr.of lines*/
  unsigned int *pl; /*pl[2i],pl[2i+1]-ends of line i*/
  unsigned int nt[2]; /*nr.of triangles*/
  unsigned int *pt; /*pt[3i],pt[3i+1],pt[3i+2]-vertices of triangles*/
  float *ct; /*colours of triangles*/
} mod2d; /*2d model composed of lines and triangles*/

unsigned int RGLOB_nrmd2d[2]={0,0}; /*nr.of models*/
mod2d *RGLOB_mod2d;


void define_chr(Uint32 *chr){
  int i;
  for(i=0;i<128;i++){chr[i]=0;}
  chr['a']=14728751; chr['b']=554584668; chr['c']=15221262; chr['d']=69683790; chr['e']=15268366;
  chr['f']=210006280; chr['g']=3236514242u; chr['h']=554584658; chr['i']=134615182; chr['j']=3288533074u;
  chr['k']=554328722; chr['l']=277094662; chr['m']=27973301; chr['n']=29968978; chr['o']=13191756;
  chr['p']=2177452944u; chr['q']=1089030594; chr['r']=15221264; chr['s']=15218750; chr['t']=277750022;
  chr['u']=19483214; chr['v']=18393412; chr['w']=18536106; chr['x']=18157905; chr['y']=3240708546u;
  chr['z']=32575775; chr[' ']=0;
  chr['A']=488177201; chr['B']=1025459774; chr['C']=488129070; chr['D']=958973532; chr['E']=1057964575;
  chr['F']=1057520144; chr['G']=520637999; chr['H']=589284913; chr['I']=474091662; chr['J']=237046348;
  chr['K']=625758801; chr['L']=554189343; chr['M']=599447089; chr['N']=597347889; chr['O']=488162862;
  chr['P']=1025047056; chr['Q']=488167023; chr['R']=1025047121; chr['S']=486999598; chr['T']=1044516996;
  chr['U']=588826158; chr['V']=588818756; chr['W']=588961450; chr['X']=581046609; chr['Y']=588583044;
  chr['Z']=1042419999;
  chr['0']=488232750; chr['1']=147460238; chr['2']=487657759; chr['3']=1041434158; chr['4']=73747426;
  chr['5']=1057948734; chr['6']=487540270; chr['7']=1041305872; chr['8']=488064558; chr['9']=488160302;
  chr['~']=444416; chr['`']=272629760; chr['!']=138547204; chr['@']=488365582; chr['#']=11512810;
  chr['$']=3438950590u; chr['%']=27070835; chr['^']=145260544; chr['&']=289691213; chr['*']=156718208;
  chr['(']=71438466; chr[')']=272765064; chr['-']=31744; chr['_']=31; chr['+']=4357252;
  chr['=']=1016800; chr['|']=138547332; chr['\\']=17043521; chr['[']=478421262; chr[']']=471926862;
  chr['{']=210002182; chr['}']=404819020; chr[':']=262152; chr[';']=2147745800u; chr['\"']=346030080;
  chr['\'']=138412032; chr['<']=2236546; chr['>']=8521864; chr[',']=2147483912u; chr['.']=396;
  chr['?']=487657476; chr['/']=1118480;
}


int initgraph(unsigned int width,unsigned int height,int fullscr,const char *title){
  unsigned int i;

if((BITSPERPIXEL!=16)&&(BITSPERPIXEL!=32)){
  printf("initgraph(): Edit file 'grconfig.h' and replace '#define BITSPERPIXEL %d'\r\n",BITSPERPIXEL);
  printf("with '#define BITSPERPIXEL 16 or #define BITSPERPIXEL 32'\r\n"); exit(1);
}

if(RGLOB_initgr==1){
  printf("initgraph(): Graphics already initialized\r\n"); return 0;
}

if(!(RGLOB_chrset=(Uint32 *)malloc(128*sizeof(Uint32)))){printf("initgraph(): Out of memory\r\n"); exit(1);}
define_chr(RGLOB_chrset);
RGLOB_csize[0]=2; RGLOB_csize[1]=2;

if((width&1)!=0){width++;}
if((height&1)!=0){height++;}
RGLOB_width=width; RGLOB_height=height;
RGLOB_xcr=0; RGLOB_ycr=0;

if(!(RGLOB_lpinc=(void **)malloc(height*sizeof(void *)))){printf("initgraph(): Out of memory\r\n"); exit(1);}
if(!(RGLOB_lpdec=(void **)malloc(height*sizeof(void *)))){printf("initgraph(): Out of memory\r\n"); exit(1);}


#if SOUND==1

/* Allocate a desired SDL_AudioSpec */
RGLOB_desired = (SDL_AudioSpec *)malloc(sizeof(SDL_AudioSpec));
/* Allocate space for the obtained SDL_AudioSpec */
RGLOB_obtained = (SDL_AudioSpec *)malloc(sizeof(SDL_AudioSpec));
/* 22050Hz - FM Radio quality */
RGLOB_desired->freq=22050;
/* 16-bit signed audio */
RGLOB_desired->format=AUDIO_S16;
/* Mono */
RGLOB_desired->channels=1;
/* Large audio buffer reduces risk of dropouts but increases response time */
RGLOB_desired->samples=1024;
/* Our callback function */
RGLOB_desired->callback=my_audio_callback;
RGLOB_desired->userdata=RGLOB_userdata;

for(i=0;i<20;i++){RGLOB_userdata[i]=0.0f;}

#endif


#if SDLVER==1

/*Initialize SDL*/
if(SDL_Init(SDL_INIT_VIDEO)<0){printf("Couldn't initialize SDL: %s\n",SDL_GetError()); SDL_Quit(); exit(1);}
/*Initialize display*/
if(fullscr==1){
  RGLOB_screen=SDL_SetVideoMode(width,height,BITSPERPIXEL,SDL_FULLSCREEN);
}else{
  RGLOB_screen=SDL_SetVideoMode(width,height,BITSPERPIXEL,SDL_SWSURFACE);
}
if(RGLOB_screen==NULL){printf("Couldn't set requested video mode: %s\n",SDL_GetError()); SDL_Quit(); exit(1);}
printf("Set %dx%dx%d\n",(RGLOB_screen->pitch)/(RGLOB_screen->format->BytesPerPixel),height,RGLOB_screen->format->BitsPerPixel);
/*Set window title*/
SDL_WM_SetCaption(title,title);
/* Enable Unicode translation */
SDL_EnableUNICODE(1);
/*SDL initialized*/

for(i=0;i<height;i++){
  RGLOB_lpinc[i]=(Uint8 *)RGLOB_screen->pixels+i*RGLOB_screen->pitch;
  RGLOB_lpdec[i]=(Uint8 *)RGLOB_screen->pixels+(height-i-1)*RGLOB_screen->pitch;
}
RGLOB_bpp=RGLOB_screen->format->BytesPerPixel;

/*Lock screen*/
if(SDL_MUSTLOCK(RGLOB_screen)){
  if(SDL_LockSurface(RGLOB_screen)<0){
    printf("initgraph(): Can't lock screen: %s\n", SDL_GetError()); SDL_Quit(); exit(1);
  }
}

#elif SDLVER==2

if(fullscr==1){printf("initgraph(): fullscr=1\r\nbut full screen not available with SDL2.\r\n");}

SDL_SetHint("SDL_FRAMEBUFFER_ACCELERATION","0");
/*Initialize SDL*/
if(SDL_Init(SDL_INIT_VIDEO)<0){printf("Couldn't initialize SDL: %s\n",SDL_GetError()); SDL_Quit(); exit(1);}
/*Initialize display*/
RGLOB_window = SDL_CreateWindow(title,SDL_WINDOWPOS_CENTERED,SDL_WINDOWPOS_CENTERED,width,height,SDL_WINDOW_SHOWN);
if(RGLOB_window==NULL){
  printf("Couldn't create window: %s\n",SDL_GetError()); SDL_Quit(); exit(1);
}
RGLOB_screen=SDL_GetWindowSurface(RGLOB_window);
if(RGLOB_screen==NULL){printf("Window surface not found\n"); SDL_Quit(); exit(1);}
printf("Set %dx%dx%d\n",(RGLOB_screen->pitch)/(RGLOB_screen->format->BytesPerPixel),height,RGLOB_screen->format->BitsPerPixel);

for(i=0;i<height;i++){
  RGLOB_lpinc[i]=(Uint8 *)RGLOB_screen->pixels+i*RGLOB_screen->pitch;
  RGLOB_lpdec[i]=(Uint8 *)RGLOB_screen->pixels+(height-i-1)*RGLOB_screen->pitch;
}
RGLOB_bpp=RGLOB_screen->format->BytesPerPixel;

/*Lock screen*/
if(SDL_MUSTLOCK(RGLOB_screen)){
  if(SDL_LockSurface(RGLOB_screen)<0){
    printf("initgraph: Can't lock screen: %s\n", SDL_GetError()); SDL_Quit(); exit(1);
  }
}

#else

if(!(RGLOB_screen=(Uint32 *)malloc(width*height*sizeof(Uint32)))){
  printf("Couldn't allocate memory for the image\r\n"); exit(1);
}

for(i=0;i<height;i++){
  RGLOB_lpinc[i]=RGLOB_screen+i*width;
  RGLOB_lpdec[i]=RGLOB_screen+(height-i-1)*width;
}
RGLOB_bpp=4;

#endif

if((RGLOB_bpp*8)!=BITSPERPIXEL){
  printf("initgraph(): For the image to be displayed correctly, edit file 'grconfig.h'\r\n");
  printf("and replace '#define BITSPERPIXEL %d' with '#define BITSPERPIXEL %d'\r\n",BITSPERPIXEL,RGLOB_bpp*8);
}

if((RGLOB_bpp*8)<BITSPERPIXEL){printf("BITSPERPIXEL too high!\r\n"); exit(1);}

#if SOUND==1

/* Open the audio device */
if (SDL_OpenAudio(RGLOB_desired,RGLOB_obtained)<0){
  fprintf(stderr,"Couldn't open audio: %s\n",SDL_GetError());
  exit(-1);
}
/* desired spec is no longer needed */
free(RGLOB_desired);
/* Start playing */
SDL_PauseAudio(0);

#endif

RGLOB_initgr=1;
setcolor(255,255,255); setbgcolor(0,0,127); clearscreen(); display();
return 1;
}


void display(){
  #if SDLVER==1
    if(RGLOB_initgr==0){printf("display(): Screen not initialized\r\n"); exit(1);}
    SDL_UpdateRect(RGLOB_screen, 0, 0, 0, 0);
  #elif SDLVER==2
    if(RGLOB_initgr==0){printf("display(): Screen not initialized\r\n"); exit(1);}
    SDL_UpdateWindowSurface(RGLOB_window);
  #else
    Uint8 b;
    Uint32 *ptr;
    unsigned long int i,j;
    FILE *f;

    if(RGLOB_initgr==0){printf("display(): Screen not initialized\r\n"); exit(1);}
    if(!(f=fopen("image.bmp","w"))){printf("display(): could not open 'image.bmp'"); return;}

    RGLOB_bmp[18]=RGLOB_width&255;
    RGLOB_bmp[19]=(RGLOB_width>>8)&255;
    RGLOB_bmp[22]=RGLOB_height&255;
    RGLOB_bmp[23]=(RGLOB_height>>8)&255;

    for(i=0;i<54;i++){putc(RGLOB_bmp[i],f);}
    for(i=0;i<RGLOB_height;i++){
      for(j=0;j<RGLOB_width;j++){
        ptr=(Uint32 *)RGLOB_lpdec[i]+j;
        b=(*ptr)&255; putc(b,f);
        b=((*ptr)>>8)&255; putc(b,f);
        b=((*ptr)>>16)&255; putc(b,f);
      }
    }

    fclose(f);
  #endif
}


void closegraph(){
  if(RGLOB_initgr==0){printf("closegraph(): Screen not initialized\r\n"); exit(1);}

  #if((SDLVER==1)||(SDLVER==2))
    /*Unlock screen*/
    if(SDL_MUSTLOCK(RGLOB_screen)){SDL_UnlockSurface(RGLOB_screen);}
    SDL_Quit();
  #else
    free(RGLOB_screen);
  #endif

  free(RGLOB_chrset); free(RGLOB_lpinc); free(RGLOB_lpdec);

  #if SOUND==1
    free(RGLOB_obtained);
  #endif

  RGLOB_initgr=0;
}


void setcolor(unsigned char red,unsigned char green,unsigned char blue){
  #if ((SDLVER==1)||(SDLVER==2))
    RGLOB_color[0]=SDL_MapRGB(RGLOB_screen->format,red,green,blue);
  #else
    RGLOB_color[0]=red; RGLOB_color[0]<<=8;
    RGLOB_color[0]|=green; RGLOB_color[0]<<=8;
    RGLOB_color[0]|=blue;
  #endif
}


void setcolrnb(float f){
  unsigned char red,green,blue;

  f*=3.0f;

  if(f<0.0f){f=0.0f;}else{if(f>3.0f){f=3.0f;}}
  if(f<=1.0){
    red=255; blue=0; green=(int)(f*255.0f);
  }else{
    if(f<=2.0){
      green=255; blue=(int)((f-1.0f)*255.0f); red=255-blue;
    }else{
      blue=255; red=0; green=(int)((3.0f-f)*255.0f);
    }
  }
  setcolor(red,green,blue);
}


void setbgcolor(unsigned char red,unsigned char green,unsigned char blue){
  #if ((SDLVER==1)||(SDLVER==2))
    RGLOB_color[1]=SDL_MapRGB(RGLOB_screen->format,red,green,blue);
  #else
    RGLOB_color[1]=red; RGLOB_color[1]<<=8;
    RGLOB_color[1]|=green; RGLOB_color[1]<<=8;
    RGLOB_color[1]|=blue;
  #endif
}


void clearscreen(){
  unsigned long int a;
  #if BITSPERPIXEL==16
    Uint16 *ptr,*ptr_end;
    if(RGLOB_initgr==0){printf("clearscreen(): Screen not initialized\r\n"); exit(1);}
    ptr=(Uint16 *)RGLOB_lpinc[0];
  #else
    Uint32 *ptr,*ptr_end;
    if(RGLOB_initgr==0){printf("clearscreen(): Screen not initialized\r\n"); exit(1);}
    ptr=(Uint32 *)RGLOB_lpinc[0];
  #endif

  a=RGLOB_width*RGLOB_height;
  ptr_end=ptr+a;
  for(;ptr<ptr_end;ptr++){*ptr=RGLOB_color[1];}
}


void scroll(int dy){
  int i;
  #if BITSPERPIXEL==16
    Uint16 *ptr,*ptr2,*ptrmax,**pl;
    pl=(Uint16 **)RGLOB_lpinc;
  #else
    Uint32 *ptr,*ptr2,*ptrmax,**pl;
    pl=(Uint32 **)RGLOB_lpinc;
  #endif

  if(RGLOB_initgr==0){printf("scroll(): Screen not initialized\r\n"); exit(1);}

  if(dy<1){dy=1;}
  if(dy>(int)(RGLOB_height>>1)){dy=RGLOB_height>>1;}

  for(i=0;i<(int)RGLOB_height-dy;i++){
    ptrmax=pl[i]+RGLOB_width;
    ptr2=pl[i+dy];
    for(ptr=pl[i];ptr<ptrmax;ptr++){
      *ptr=*ptr2;
      ptr2++;
    }
  }

  for(i=RGLOB_height-dy;i<(int)RGLOB_height;i++){
    ptrmax=pl[i]+RGLOB_width;
    for(ptr=pl[i];ptr<ptrmax;ptr++){*ptr=RGLOB_color[1];}
  }
}


void dpoint(int x,int y){
  if(RGLOB_initgr==0){printf("dpoint(): Screen not initialized\r\n"); exit(1);}

  if((y>=0)&&(y<(int)RGLOB_height)&&(x>=0)&&(x<(int)RGLOB_width)){
    #if BITSPERPIXEL==16
      *((Uint16 *)RGLOB_lpdec[y]+x)=RGLOB_color[0];
    #else
      *((Uint32 *)RGLOB_lpdec[y]+x)=RGLOB_color[0];
    #endif
    RGLOB_xcr=x; RGLOB_ycr=y;
  }
}


/*Bresenham's line algorithm*/
void dlinerel(int x,int y){
  int dx,dy,xc,yc,x1,y1,a;
  #if BITSPERPIXEL==16
    Uint16 **pl;
    pl=(Uint16 **)RGLOB_lpdec;
  #else
    Uint32 **pl;
    pl=(Uint32 **)RGLOB_lpdec;
  #endif

if(RGLOB_initgr==0){printf("dlinerel(): Screen not initialized\r\n"); exit(1);}

xc=RGLOB_xcr; yc=RGLOB_ycr; x1=xc+x; y1=yc+y;

if((x1>=0)&&(x1<(int)RGLOB_width)&&(y1>=0)&&(y1<(int)RGLOB_height)){
  RGLOB_xcr=x1; RGLOB_ycr=y1;
  if(x>0){dx=1;}else{dx=-1; x=-x;}
  if(y>0){dy=1;}else{dy=-1; y=-y;}
  if(x>y){
    a=x; x+=x; y+=y;
    while(xc!=x1){
      xc+=dx; a-=y;
      if(a<=0){yc+=dy; a+=x;}
      *(pl[yc]+xc)=RGLOB_color[0];
    }
  }else{
    a=y; x+=x; y+=y;
    while(yc!=y1){
      yc+=dy; a-=x;
      if(a<=0){xc+=dx; a+=y;}
      *(pl[yc]+xc)=RGLOB_color[0];
    }
  }
}
}


/*Horizontal,vertical or diagonal line*/
void dlrspec(int dx,int dy,int n){
  int xc,yc,x1,y1;
  #if BITSPERPIXEL==16
    Uint16 **pl;
    pl=(Uint16 **)RGLOB_lpdec;
  #else
    Uint32 **pl;
    pl=(Uint32 **)RGLOB_lpdec;
  #endif

if(RGLOB_initgr==0){printf("dlrspec(): Screen not initialized\r\n"); exit(1);}

xc=RGLOB_xcr; x1=xc+dx*n; yc=RGLOB_ycr; y1=yc+dy*n;

if((x1>=0)&&(x1<(int)RGLOB_width)&&(y1>=0)&&(y1<(int)RGLOB_height)){
  RGLOB_xcr=x1; RGLOB_ycr=y1;
  if(dx!=0){
    while(xc!=x1){
      xc+=dx; yc+=dy; *(pl[yc]+xc)=RGLOB_color[0];
    }
  }else{
    if(dy!=0){
      while(yc!=y1){
        xc+=dx; yc+=dy; *(pl[yc]+xc)=RGLOB_color[0];
      }
    }
  }
}
}


void drectangle (int x,int y){
  int xc,yc,x1,y1;
  #if BITSPERPIXEL==16
    Uint16 *ptr,*ptrmax,**pl;
    pl=(Uint16 **)RGLOB_lpdec;
  #else
    Uint32 *ptr,*ptrmax,**pl;
    pl=(Uint32 **)RGLOB_lpdec;
  #endif

  if(RGLOB_initgr==0){printf("drectangle(): Screen not initialized\r\n"); exit(1);}

  xc=RGLOB_xcr; yc=RGLOB_ycr; x1=xc+x; y1=yc+y;

  if((x1>=0)&&(x1<(int)RGLOB_width)&&(y1>=0)&&(y1<(int)RGLOB_height)){
    RGLOB_xcr=x1; RGLOB_ycr=y1;
    if(xc>x1){x=xc; xc=x1; x1=x;}
    if(yc>y1){y=yc; yc=y1; y1=y;}

    for(;yc<y1;yc++){
      ptrmax=pl[yc]+x1;
      for(ptr=pl[yc]+xc;ptr<ptrmax;ptr++){*ptr=RGLOB_color[0];}
    }
  }
}


void setchrsize(int i,int j){
  if(i<1){i=1;}else{if(i>8){i=8;}}
  if(j<1){j=1;}else{if(j>8){j=8;}}
  RGLOB_csize[0]=i; /*horizontal*/
  RGLOB_csize[1]=j; /*vertical*/
}


/*bgr=1-draw background;0-don't*/
void dchar(char c,int x,int y,int bgr){
  int i,j,k,l,sh,sv,s5,s6;
  Uint32 a;
  #if BITSPERPIXEL==16
    Uint16 *ptr,**pl;
    pl=(Uint16 **)RGLOB_lpdec;
  #else
    Uint32 *ptr,**pl;
    pl=(Uint32 **)RGLOB_lpdec;
  #endif

if(RGLOB_initgr==0){printf("dchar(): Screen not initialized\r\n"); exit(1);}

sh=RGLOB_csize[0]; sv=RGLOB_csize[1];
if((c<0)||(c>126)){printf("dchar(%d): Unknown character\r\n",c); return;}

s5=sh*7; s6=sv*8;

if((x<0)||(y<0)||(x>(int)(RGLOB_width-s5))||(y>(int)(RGLOB_height-s6))){
  printf("dchar(): Character outside of screen\r\n"); return;
}

if(bgr){
  for(i=0;i<s6;i++){
    ptr=pl[y+i]+x;
    for(j=0;j<s5;j++){
      *ptr=RGLOB_color[1]; ptr++;
    }
  }
}

a=*(RGLOB_chrset+c);
s5=sh*5; s6=sv*6;
for(i=sv;i<=s6;i+=sv){
  for(j=s5;j>=sh;j-=sh){
    if(a&1){
      for(k=0;k<sv;k++){
        ptr=pl[y+i+k]+x+j;
        for(l=0;l<sh;l++){
          *ptr=RGLOB_color[0]; ptr++;
        }
      }
    }
    a>>=1;
  }
}

switch(a){
  case 0: break;
  case 1: s5=sh*4; s6=s5; break; /*letter 'q'*/
  case 2: s5=sh; s6=s5; break; /*letter 'p'*/
  case 3: s5=sh*2; s6=s5+sh; break; /*letters 'g','j','i'*/
  default: break;
}

if(a>0){
  for(j=s5;j<=s6;j+=sh){
    for(k=0;k<sv;k++){
      ptr=pl[y+k]+x+j;
      for(l=0;l<sh;l++){
        *ptr=RGLOB_color[0]; ptr++;
      }
    }
  }
}

RGLOB_xcr=x; RGLOB_ycr=y;
}


void dtext(char *tx,int x,int y,int bgr){
  int sh,sv,i;

  sh=7*RGLOB_csize[0]; sv=8*RGLOB_csize[1]; i=0;
  while((tx[i]!='\0')&&(i<1024)){
    if((x>((int)RGLOB_width-sh))||(tx[i]=='\n')){x=0; y-=sv;}
    if((*(RGLOB_chrset+tx[i])!=0)||(tx[i]==' ')){dchar(tx[i],x,y,bgr); x+=sh;}
    i++;
  }
  if(x>((int)RGLOB_width-sh)){x=0; y-=sv;}
  RGLOB_xcr=x; RGLOB_ycr=y;
}


void dtextscroll(char *tx,int x,int y,int bgr){
  int sh,sv,i;

  sh=7*RGLOB_csize[0]; sv=8*RGLOB_csize[1]; i=0;
  while((tx[i]!='\0')&&(i<1024)){
    if((x>((int)RGLOB_width-sh))||(tx[i]=='\n')){x=0; scroll(sv);}
    if((*(RGLOB_chrset+tx[i])!=0)||(tx[i]==' ')){dchar(tx[i],x,y,bgr); x+=sh;}
    i++;
  }
  if(x>((int)RGLOB_width-sh)){x=0; scroll(sv);}
  RGLOB_xcr=x; RGLOB_ycr=y;
}


void dlineabs(float x1,float y1,float x2,float y2){
  float x[4],h,m;
  unsigned int i,i1,i2,i3;

  x[0]=x1; x[1]=x2; x[2]=y1; x[3]=y2;
  m=0.1;

for(i=0;i<4;i++){
  i1=(5-i)&3; i2=(2+i)&3; i3=3-i;
  h=(i<2) ? (RGLOB_width-0.6f) : (RGLOB_height-0.6f);

  if(x[i]<m){
    if(x[i1]>m+0.5f){
      x[i2]+=(m-x[i])*(x[i3]-x[i2])/(x[i1]-x[i]);
      x[i]=m;
    }else{return;}
  }

  if(x[i]>h){
    if(x[i1]<h-0.5f){
      x[i2]+=(h-x[i])*(x[i3]-x[i2])/(x[i1]-x[i]);
      x[i]=h;
    }else{return;}
  }
}

  h=(x[1]>=x[0]) ? ((int)(x[1]-x[0])) : (-(int)(x[0]-x[1]));
  m=(x[3]>=x[2]) ? ((int)(x[3]-x[2])) : (-(int)(x[2]-x[3]));

  dpoint((int)x[0],(int)x[2]); dlinerel((int)h,(int)m);
}


/*y is x and x is y;prototype is void dtriangle(float x1,float y1,float x2,float y2,float x3,float y3);*/
void dtriangle(float y1,float x1,float y2,float x2,float y3,float x3){
  int i,imin,imax,jmin,jmax;
  float tmp,slope2,slope3,slope4,xcr,ystart,yend,thres=1.0e-5f;
  #if BITSPERPIXEL==16
    Uint16 *ptr,*ptrmax,**pl;
    pl=(Uint16 **)RGLOB_lpdec;
  #else
    Uint32 *ptr,*ptrmax,**pl;
    pl=(Uint32 **)RGLOB_lpdec;
  #endif

if(RGLOB_initgr==0){printf("dtriangle(): Screen not initialized\r\n"); exit(1);}

if(x2>x3){tmp=y2; y2=y3; y3=tmp; tmp=x2; x2=x3; x3=tmp;}
if(x1>x2){
  tmp=y1; y1=y2; y2=tmp; tmp=x1; x1=x2; x2=tmp;
  if(x2>x3){tmp=y2; y2=y3; y3=tmp; tmp=x2; x2=x3; x3=tmp;}
} /*ordered triangle vertices*/

imax=(int)x3; imin=(int)(x1+0.99999f);

if(imin<0){imin=0;}
if(imax>((int)RGLOB_height-1)){imax=(int)RGLOB_height-1;} /*found triangle limits*/

if((x2-x1)>thres){slope2=(y2-y1)/(x2-x1);}else{slope2=(y2-y1)/thres;}
if((x3-x1)>thres){slope3=(y3-y1)/(x3-x1);}else{slope3=(y3-y1)/thres;}
if((x3-x2)>thres){slope4=(y3-y2)/(x3-x2);}else{slope4=(y3-y2)/thres;}

for(i=imin;i<=imax;i++){
  xcr=i;
  if(xcr<x2){
    ystart=(slope2*(xcr-x1)+y1);
    yend=(slope3*(xcr-x1)+y1);}
  else{
    ystart=(slope4*(xcr-x2)+y2);
    yend=(slope3*(xcr-x1)+y1);}
  if(yend<ystart){tmp=ystart;ystart=yend;yend=tmp;}

  jmin=(int)(ystart+0.99f); jmax=(int)yend;
  if(jmin<0){jmin=0;}
  if(jmax>((int)RGLOB_width-1)){jmax=(int)RGLOB_width-1;}

  ptrmax=pl[i]+jmax;

  for(ptr=pl[i]+jmin;ptr<=ptrmax;ptr++){
    *ptr=RGLOB_color[0];
  }
}
}


void dcircle(float x0,float y0,float r){
  int i,nseg; /*nseg-nr.segments per octant*/
  float s[2],c[2],arc,x,x2;

if(r<0){r=-r;}
nseg=(int)(0.05f*r)+1; if(nseg>7){nseg=7;}
arc=0.789f/(float)nseg; x=0.0f; s[0]=0.0f; c[0]=r;
x0=((int)x0)+0.5f; y0=((int)y0)+0.5f;

for(i=0;i<nseg;i++){
  x+=arc; x2=x*x;
  s[1]=r*x*(1.0f-x2*(0.1667f-x2*0.00833f)); /*r sin(x)*/
  c[1]=r*(1.0f-x2*(0.5f-x2*0.04167f)); /*r cos(x)*/
  dlineabs(x0+c[0],y0+s[0],x0+c[1],y0+s[1]);
  dlineabs(x0-c[0],y0+s[0],x0-c[1],y0+s[1]);
  dlineabs(x0+c[0],y0-s[0],x0+c[1],y0-s[1]);
  dlineabs(x0-c[0],y0-s[0],x0-c[1],y0-s[1]);
  dlineabs(x0+s[0],y0+c[0],x0+s[1],y0+c[1]);
  dlineabs(x0-s[0],y0+c[0],x0-s[1],y0+c[1]);
  dlineabs(x0+s[0],y0-c[0],x0+s[1],y0-c[1]);
  dlineabs(x0-s[0],y0-c[0],x0-s[1],y0-c[1]);
  s[0]=s[1]; c[0]=c[1];
}
}


void ddisc(float x0f,float y0f,float rf){
  int i,i2,x0,y0,r,r2,xmin,xmax,ymin,ymax,ym,dx,dx2;
  #if BITSPERPIXEL==16
    Uint16 *ptr,*ptrmax,**pl;
    pl=(Uint16 **)RGLOB_lpdec;
  #else
    Uint32 *ptr,*ptrmax,**pl;
    pl=(Uint32 **)RGLOB_lpdec;
  #endif

if(RGLOB_initgr==0){printf("ddisc(): Screen not initialized\r\n"); exit(1);}
x0=(int)x0f; y0=(int)y0f; r=(int)rf; r2=r*r;

ymin=y0-r; if(ymin>=(int)RGLOB_height){return;}else{if(ymin<0){ymin=0;}}
ymax=y0+r; if(ymax<0){return;}else{if(ymax>=(int)RGLOB_height){ymax=RGLOB_height-1;}}

xmin=x0-r; if(xmin>=(int)RGLOB_width){return;}else{if(xmin<0){xmin=0;}}
xmax=x0+r; if(xmax<0){return;}else{if(xmax>=(int)RGLOB_width){xmax=RGLOB_width-1;}}

dx=r; dx2=r2;
if((ymax-y0)>(y0-ymin)){ym=ymax-y0;}else{ym=y0-ymin;}

if((y0<=ymax)&&(y0>=ymin)){for(i=xmin;i<=xmax;i++){*(pl[y0]+i)=RGLOB_color[0];}}

i2=0;
for(i=1;i<=ym;i++){
  i2+=(i+i-1);
  while(dx2>(r2-i2)){dx2-=dx; dx--; dx2-=dx;}
  xmin=x0-dx; if(xmin<0){xmin=0;}
  xmax=x0+dx; if(xmax>=(int)RGLOB_width){xmax=RGLOB_width-1;}
  if(((y0+i)<=ymax)&&((y0+i)>=ymin)){
    ptrmax=pl[y0+i]+xmax;
    for(ptr=pl[y0+i]+xmin;ptr<=ptrmax;ptr++){*ptr=RGLOB_color[0];}
  }
  if(((y0-i)>=ymin)&&((y0-i)<=ymax)){
    ptrmax=pl[y0-i]+xmax;
    for(ptr=pl[y0-i]+xmin;ptr<=ptrmax;ptr++){*ptr=RGLOB_color[0];}
  }
}
}


/*like dpoint(),but without the point*/
void setpos(int x,int y){
  if(RGLOB_initgr==0){printf("setpos(): Screen not initialized\r\n"); exit(1);}
  if((y>=0)&&(y<(int)RGLOB_height)&&(x>=0)&&(x<(int)RGLOB_width)){RGLOB_xcr=x; RGLOB_ycr=y;}
}


void printw(const char *format, ...){
  char sir[256];
  va_list argp;

  va_start(argp,format);
  vsprintf(sir,format,argp);
  dtext(sir,RGLOB_xcr,RGLOB_ycr,1);
  va_end(argp);
}


void printws(const char *format, ...){
  char sir[256];
  va_list argp;

  if((RGLOB_ycr!=0)||((RGLOB_xcr%(7*RGLOB_csize[0]))!=0)){
    RGLOB_xcr=0; RGLOB_ycr=0; scroll(8*RGLOB_csize[1]);
  }

  va_start(argp,format);
  vsprintf(sir,format,argp);
  dtextscroll(sir,RGLOB_xcr,RGLOB_ycr,1);
  va_end(argp);
}


#if ((SDLVER==1)||(SDLVER==2))

void pause(int p){
  SDL_Delay(p);
}


char getch_graph(){
  char rez=0;
  int cr;

dchar('_',RGLOB_xcr,RGLOB_ycr,1); display();

cr=0;

while(!rez){ SDL_Delay(10); cr++; if(cr==50){display(); cr=0;}
while(SDL_PollEvent(&RGLOB_event)){cr=0;
switch(RGLOB_event.type){

case SDL_KEYDOWN:
  switch(RGLOB_event.key.keysym.sym){
    case SDLK_a: rez='a'; break;
    case SDLK_b: rez='b'; break;
    case SDLK_c: rez='c'; break;
    case SDLK_d: rez='d'; break;
    case SDLK_e: rez='e'; break;
    case SDLK_f: rez='f'; break;
    case SDLK_g: rez='g'; break;
    case SDLK_h: rez='h'; break;
    case SDLK_i: rez='i'; break;
    case SDLK_j: rez='j'; break;
    case SDLK_k: rez='k'; break;
    case SDLK_l: rez='l'; break;
    case SDLK_m: rez='m'; break;
    case SDLK_n: rez='n'; break;
    case SDLK_o: rez='o'; break;
    case SDLK_p: rez='p'; break;
    case SDLK_q: rez='q'; break;
    case SDLK_r: rez='r'; break;
    case SDLK_s: rez='s'; break;
    case SDLK_t: rez='t'; break;
    case SDLK_u: rez='u'; break;
    case SDLK_v: rez='v'; break;
    case SDLK_w: rez='w'; break;
    case SDLK_x: rez='x'; break;
    case SDLK_y: rez='y'; break;
    case SDLK_z: rez='z'; break;
    case SDLK_0: rez='0'; break;
    case SDLK_1: rez='1'; break;
    case SDLK_2: rez='2'; break;
    case SDLK_3: rez='3'; break;
    case SDLK_4: rez='4'; break;
    case SDLK_5: rez='5'; break;
    case SDLK_6: rez='6'; break;
    case SDLK_7: rez='7'; break;
    case SDLK_8: rez='8'; break;
    case SDLK_9: rez='9'; break;
  #if SDLVER==1
    case SDLK_KP0: rez='0'; break;
    case SDLK_KP1: rez='1'; break;
    case SDLK_KP2: rez='2'; break;
    case SDLK_KP3: rez='3'; break;
    case SDLK_KP4: rez='4'; break;
    case SDLK_KP5: rez='5'; break;
    case SDLK_KP6: rez='6'; break;
    case SDLK_KP7: rez='7'; break;
    case SDLK_KP8: rez='8'; break;
    case SDLK_KP9: rez='9'; break;
  #else
    case SDLK_KP_0: rez='0'; break;
    case SDLK_KP_1: rez='1'; break;
    case SDLK_KP_2: rez='2'; break;
    case SDLK_KP_3: rez='3'; break;
    case SDLK_KP_4: rez='4'; break;
    case SDLK_KP_5: rez='5'; break;
    case SDLK_KP_6: rez='6'; break;
    case SDLK_KP_7: rez='7'; break;
    case SDLK_KP_8: rez='8'; break;
    case SDLK_KP_9: rez='9'; break;
  #endif
    case SDLK_SPACE: rez=' '; break;
    case SDLK_PERIOD:
    case SDLK_KP_PERIOD:
    case SDLK_COMMA: rez='.'; break;
    case SDLK_MINUS:
    case SDLK_KP_MINUS: rez='-'; break;
    case SDLK_BACKSPACE:
    case SDLK_DELETE: rez=1; break;
    case SDLK_RETURN:
    case SDLK_KP_ENTER: rez=2; break;

    case SDLK_ESCAPE: rez=3; break;

    default: break;

} break;

case SDL_KEYUP:
  switch(RGLOB_event.key.keysym.sym){
    default: break;
} break;

case SDL_QUIT: closegraph(); exit(1);

}}}

if(rez>3){dchar(rez,RGLOB_xcr,RGLOB_ycr,1);}

return rez;
}


char getch(){
  scroll(8*RGLOB_csize[1]); RGLOB_xcr=0; RGLOB_ycr=0;
  return getch_graph();
}


void vxscanw(const char *format,va_list argp){
  char sir[256];
  unsigned int i=0,sem=0,sh;

sh=7*RGLOB_csize[0];

while(!sem){
  if(RGLOB_xcr<=(int)(RGLOB_width-sh)){
    sir[i]=getch_graph();
  }else{
    sem=1; sir[i]='\0';
  }
  switch(sir[i]){
    case 1: if(i>=1){i--; dchar(' ',RGLOB_xcr,RGLOB_ycr,1); RGLOB_xcr-=sh;} break;
    case 2: sem=1; sir[i]='\0'; dchar(' ',RGLOB_xcr,RGLOB_ycr,1); break;
    case 3: while(i>=1){i--; dchar(' ',RGLOB_xcr,RGLOB_ycr,1); RGLOB_xcr-=sh;}
            sem=1; sir[0]='0'; sir[1]='\0';
            dchar('0',RGLOB_xcr,RGLOB_ycr,1); break;
    default: i++; RGLOB_xcr+=sh; break;
  }
}
vsscanf(sir,format,argp);
}


void scanw(const char *format, ...){
  va_list argp;

  va_start(argp,format);
  vxscanw(format,argp);
  va_end(argp);
}


void scanws(const char *format, ...){
  va_list argp;

  if((RGLOB_ycr!=0)||((RGLOB_xcr%(7*RGLOB_csize[0]))!=0)){
    RGLOB_xcr=0; RGLOB_ycr=0; scroll(8*RGLOB_csize[1]);
  }

  va_start(argp,format);
  vxscanw(format,argp);
  va_end(argp);

  scroll(8*RGLOB_csize[1]); RGLOB_xcr=0;
}


#else

void pause(int p){printf("pause() requires SDL\r\n");}
char getch_graph(){printf("getch_graph() requires SDL\r\n"); return 0;}
char getch(){printf("getch() requires SDL\r\n"); return 0;}
void scanw(const char *format, ...){printf("scanw() requires SDL\r\n");}
void scanws(const char *format, ...){printf("scanws() requires SDL\r\n");}

#endif


int addmod2d(){
  unsigned int i,m,n=8;
  mod2d *m2d;

i=(RGLOB_nrmd2d[0])++;
m=RGLOB_nrmd2d[1];

if(i>=m){
  m+=n;
  RGLOB_nrmd2d[1]=m;
  if(m==n){
    if(!(RGLOB_mod2d=(mod2d *)malloc(m*sizeof(mod2d)))){printf("addmod2d(): Out of memory\r\n"); exit(1);}
  }else{
    if(!(RGLOB_mod2d=(mod2d *)realloc(RGLOB_mod2d,m*sizeof(mod2d)))){printf("addmod2d(): Out of memory\r\n"); exit(1);}
  }
}

m2d=&(RGLOB_mod2d[i]);

m2d->np[0]=0; m2d->np[1]=n;
if(!(m2d->x=(float *)malloc(n*2*sizeof(float)))){printf("addmod2d(): Out of memory\r\n"); exit(1);}

m2d->nl[0]=0; m2d->nl[1]=n;
if(!(m2d->pl=(unsigned int *)malloc(n*2*sizeof(int)))){printf("addmod2d(): Out of memory\r\n"); exit(1);}

m2d->nt[0]=0; m2d->nt[1]=n;
if(!(m2d->pt=(unsigned int *)malloc(n*3*sizeof(int)))){printf("addmod2d(): Out of memory\r\n"); exit(1);}
if(!(m2d->ct=(float *)malloc(n*sizeof(float)))){printf("addmod2d(): Out of memory\r\n"); exit(1);}

return i;
}


void freemod2d(){
  unsigned int i;
  mod2d *m2d;
if(RGLOB_nrmd2d[0]>0){
  for(i=0;i<RGLOB_nrmd2d[0];i++){
    m2d=RGLOB_mod2d+i;
    free(m2d->x); free(m2d->pl); free(m2d->pt); free(m2d->ct);
  }
  free(RGLOB_mod2d);
  RGLOB_nrmd2d[0]=0; RGLOB_nrmd2d[1]=0;
}
}


void addpoint2d(float x,float y){
  int nmod,i,m,n=8;
  mod2d *m2d;

  nmod=RGLOB_nrmd2d[0]-1;

  if(nmod==-1){printf("addpoint2d(): Call addmod2d() first\r\n"); exit(1);}

  m2d=&(RGLOB_mod2d[nmod]);
  i=(m2d->np[0])++; m=m2d->np[1];
  if(i>=m){
    m+=n;
    m2d->np[1]=m;
    if(!(m2d->x=(float *)realloc(m2d->x,m*2*sizeof(float)))){printf("addpoint2d(): Out of memory\r\n"); exit(1);}
  }
  m=2*i; m2d->x[m]=x; m2d->x[m+1]=y;
}


void addline2d(unsigned int p1,unsigned int p2){
  int nmod,i,m,n=8;
  mod2d *m2d;

  nmod=RGLOB_nrmd2d[0]-1;

  if(nmod==-1){printf("addline2d(): Call addmod2d() first\r\n"); exit(1);}

  m2d=&(RGLOB_mod2d[nmod]);

  if((p1>=(m2d->np[0]))||(p2>=(m2d->np[0]))){printf("addline2d(): model %d has only %d points\r\n",nmod,m2d->np[0]); exit(1);}

  i=(m2d->nl[0])++; m=m2d->nl[1];
  if(i>=m){
    m+=n;
    m2d->nl[1]=m;
    if(!(m2d->pl=(unsigned int *)realloc(m2d->pl,m*2*sizeof(int)))){printf("addline2d(): Out of memory\r\n"); exit(1);}
  }
  m=2*i; m2d->pl[m]=p1; m2d->pl[m+1]=p2;
}


void addtriangle2d(unsigned int p1,unsigned int p2,unsigned int p3,float col){
  int nmod,i,m,n=8;
  mod2d *m2d;

  nmod=RGLOB_nrmd2d[0]-1;

  if(nmod==-1){printf("addtriangle2d(): Call addmod2d() first\r\n"); exit(1);}

  m2d=&(RGLOB_mod2d[nmod]);

  if((p1>=(m2d->np[0]))||(p2>=(m2d->np[0]))||(p3>=(m2d->np[0]))){printf("addtriangle2d(): model %d has only %d points\r\n",nmod,m2d->np[0]); exit(1);}

  i=(m2d->nt[0])++; m=m2d->nt[1];
  if(i>=m){
    m+=n;
    m2d->nt[1]=m;
    if(!(m2d->pt=(unsigned int *)realloc(m2d->pt,m*3*sizeof(int)))){printf("addtriangle2d(): Out of memory\r\n"); exit(1);}
    if(!(m2d->ct=(float *)realloc(m2d->ct,m*sizeof(float)))){printf("addtriangle2d(): Out of memory\r\n"); exit(1);}
  }
  m=3*i; m2d->pt[m]=p1; m2d->pt[m+1]=p2; m2d->pt[m+2]=p3; m2d->ct[i]=col;
}


void drawmod2d(unsigned int nmod,float x0,float y0,float scale,float ang){
  unsigned int i,j,n;
  Uint32 col;
  float x,y,x1,y1,x2,y2,x3,y3,c,s;
  mod2d *m2d;

if(RGLOB_initgr==0){printf("drawmod2d(): Screen not initialized\r\n"); exit(1);}
if(nmod>=RGLOB_nrmd2d[0]){printf("drawmod2d(): model %d not exist\r\n",nmod); exit(1);}

m2d=RGLOB_mod2d+nmod;
c=cos(ang); s=sin(ang);
col=RGLOB_color[0];

n=m2d->nt[0];
for(i=0;i<n;i++){
  j=i*3;
  x=scale*(m2d->x[(m2d->pt[j])<<1]);
  y=scale*(m2d->x[((m2d->pt[j])<<1)+1]);
  x1=x0+c*x-s*y;
  y1=y0+s*x+c*y;
  x=scale*(m2d->x[(m2d->pt[j+1])<<1]);
  y=scale*(m2d->x[((m2d->pt[j+1])<<1)+1]);
  x2=x0+c*x-s*y;
  y2=y0+s*x+c*y;
  x=scale*(m2d->x[(m2d->pt[j+2])<<1]);
  y=scale*(m2d->x[((m2d->pt[j+2])<<1)+1]);
  x3=x0+c*x-s*y;
  y3=y0+s*x+c*y;
  setcolrnb(m2d->ct[i]);
  dtriangle(x1,y1,x2,y2,x3,y3);
}

RGLOB_color[0]=col;

n=m2d->nl[0];
for(i=0;i<n;i++){
  j=i<<1;
  x=scale*(m2d->x[(m2d->pl[j])<<1]);
  y=scale*(m2d->x[((m2d->pl[j])<<1)+1]);
  x1=x0+c*x-s*y;
  y1=y0+s*x+c*y;
  x=scale*(m2d->x[(m2d->pl[j+1])<<1]);
  y=scale*(m2d->x[((m2d->pl[j+1])<<1)+1]);
  x2=x0+c*x-s*y;
  y2=y0+s*x+c*y;
  dlineabs(x1,y1,x2,y2);
}

}


void evmod2d(unsigned int n1,unsigned int n2,float *xmin,float *ymin,float *scale){
  unsigned int i,j,n;
  float x1,y1,x2,y2;
  mod2d *m2d;

if(RGLOB_initgr==0){printf("evmod2d(): Screen not initialized\r\n"); exit(1);}
if(n1>n2){i=n1; n1=n2; n2=i;}
if(n2>=RGLOB_nrmd2d[0]){printf("evmod2d(): model %d not exist\r\n",n2); exit(1);}

m2d=&(RGLOB_mod2d[n1]); x1=x2=m2d->x[0]; y1=y2=m2d->x[1];
for(i=n1;i<=n2;i++){
  m2d=&(RGLOB_mod2d[i]); n=m2d->np[0];
  for(j=0;j<n;j++){
    if(x1 > m2d->x[j<<1]){x1 = m2d->x[j<<1];}
    if(x2 < m2d->x[j<<1]){x2 = m2d->x[j<<1];}
    if(y1 > m2d->x[(j<<1)+1]){y1 = m2d->x[(j<<1)+1];}
    if(y2 < m2d->x[(j<<1)+1]){y2 = m2d->x[(j<<1)+1];}
  }
}

(*xmin)=x1; (*ymin)=y1;
x1=((float)RGLOB_width-20.0f)/(x2-x1);
y1=((float)RGLOB_height-20.0f)/(y2-y1);
(*scale) = (x1>y1) ? y1 : x1;
}


void drmod2dauto(unsigned int nmod){
  float xmin,ymin,scale;

  evmod2d(nmod,nmod,&xmin,&ymin,&scale);
  drawmod2d(nmod, 10.0f-xmin*scale, 10.0f-ymin*scale, scale, 0.0f);
}
